/* ***************************************************** **
   ch02_pitfalls_of_set_operations.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 2.
   Problemy związane z operacjami na zbiorach
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

-- W przeciwieństwie do innych rozdziałów, w tym kolmny są
-- formatowane ręcznie z użyciem sqlformat ansiconsole

set pagesize 80
set linesize 80
set sqlformat
alter session set nls_date_format = 'YYYY-MM-DD';

column c_id          format 99999
column customer_name format a15
column b_id          format 99999
column brewery_name  format a18
column p_id          format 9999
column product_name  format a17
column c_or_b_id     format 99999
column c_or_b_name   format a18
column ordered       format a10
column qty           format 999
column product_coll  format a40
column multiset_coll format a60
column rn            format 9

/* -----------------------------------------------------
   Przykładowe fragmenty kodu do rozdziału 2.
   ----------------------------------------------------- */

-- Listing 2.2. Dane dotyczące dwóch klientów i ich zamówień

select
   customer_id as c_id, customer_name, ordered
 , product_id  as p_id, product_name , qty
from customer_order_products
where customer_id in (50042, 50741)
order by customer_id, product_id;

-- Listing 2.3. Dane dotyczące dwóch browarów i kupowanych od nich produktów

select
   brewery_id as b_id, brewery_name
 , product_id as p_id, product_name
from brewery_products
where brewery_id in (518, 523)
order by brewery_id, product_id;

-- Listing 2.4. Konkatenacja wyników dwóch zapytań

select product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
union all
select product_id as p_id, product_name
from brewery_products
where brewery_id = 523;

-- Listing 2.5. Odmienne kolumny użyte w dwóch zapytaniach

select
   customer_id as c_or_b_id, customer_name as c_or_b_name
 , product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
union all
select
   brewery_id, brewery_name
 , product_id as p_id, product_name
from brewery_products
where brewery_id = 523;

-- Próba użycia kolejności według kolumny tabeli prowadzi do błędu ORA-00904: "PRODUCT_ID": invalid identifier

select
   customer_id as c_or_b_id, customer_name as c_or_b_name
 , product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
union all
select
   brewery_id, brewery_name
 , product_id as p_id, product_name
from brewery_products
where brewery_id = 523
order by product_id;

-- Ułożenie według aliasów kolumn działa bez problemów

select
   customer_id as c_or_b_id, customer_name as c_or_b_name
 , product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
union all
select
   brewery_id, brewery_name
 , product_id as p_id, product_name
from brewery_products
where brewery_id = 523
order by p_id;

-- Listing 2.6. Unia to prawdziwa operacja na zbiorze, która przeprowadza niejawne usunięcie duplikatów ze zbioru wynikowego

select product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
union
select product_id as p_id, product_name
from brewery_products
where brewery_id = 523
order by p_id;

-- użycie operatora intersect powoduje wygenerowanie iloczynu rekordów

select product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
intersect
select product_id as p_id, product_name
from brewery_products
where brewery_id = 523
order by p_id;

-- Minus powoduje wygenerowanie zbioru unikatowych rekordów pierwszego zapytania select, które nie znajdują się w wyniku wykonania drugiego zapytania select

select product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
minus
select product_id as p_id, product_name
from brewery_products
where brewery_id = 523
order by p_id;

-- Listing 2.7. Dane produktu klienta wyświetlone jako typ kolekcji

select
   customer_id as c_id, customer_name
 , product_coll
from customer_order_products_obj
where customer_id in (50042, 50741)
order by customer_id;

-- Listing 2.8. Unia wielozbiorów w kolekcji

select
   whitehart.product_coll
   multiset union
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Multiset union all daje taki sam wynik jak w przypadku multiset union

select
   whitehart.product_coll
   multiset union all
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Jeżeli do multiset union zostanie dodane słowo kluczowe distinct, dane wyjściowe nie będą zawierały duplikatów 

select
   whitehart.product_coll
   multiset union distinct
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- W przypadlu multiset można użyć intersect all

select
   whitehart.product_coll
   multiset intersect all
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- A także intersect distinct

select
   whitehart.product_coll
   multiset intersect distinct
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Oczywiście można również użyć except all

select
   whitehart.product_coll
   multiset except all
   hyggehumle.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Znacznie bardziej interesujące wyniki otrzymamy po odwróceniu except all

select
   hyggehumle.product_coll
   multiset except all
   whitehart.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Wynik wykonania except distinct to najpierw pobranie unikatowych wartości, a później odjęcie zbioru

select
   hyggehumle.product_coll
   multiset except distinct
   whitehart.product_coll
      as multiset_coll
from customer_order_products_obj whitehart
cross join customer_order_products_obj hyggehumle
where whitehart.customer_id = 50042
and hyggehumle.customer_id = 50741;

-- Listing 2.9. Operator minus przypomina w działaniu operator multiset except distinct 

select product_id as p_id, product_name
from customer_order_products
where customer_id = 50741
minus
select product_id as p_id, product_name
from customer_order_products
where customer_id = 50042
order by p_id;

-- Listing 2.10. Emulacja operatora minus all za pomocą operatora multiset except all 

select
   minus_all_table.id   as p_id
 , minus_all_table.name as product_name
from table(
   cast(
      multiset(
         select product_id, product_name
         from customer_order_products
         where customer_id = 50741
      )
      as id_name_coll_type
   )
   multiset except all
   cast(
      multiset(
         select product_id, product_name
         from customer_order_products
         where customer_id = 50042
      )
      as id_name_coll_type
   )
) minus_all_table
order by p_id;

-- Listing 2.11. Emulowanie operatora MINUS ALL za pomocą funkcji analitycznej row_number()

select
   product_id as p_id
 , product_name
 , row_number() over (
      partition by product_id, product_name
      order by rownum
   ) as rn
from customer_order_products
where customer_id = 50741
minus
select
   product_id as p_id
 , product_name
 , row_number() over (
      partition by product_id, product_name
      order by rownum
   ) as rn
from customer_order_products
where customer_id = 50042
order by p_id;

/* ***************************************************** */
